<?php

namespace App\Services\Sms;

use Illuminate\Support\Facades\Http;

/**
 * Class NetgsmSaglayici
 * Netgsm API'si ile SMS gönderimini gerçekleştirir.
 */
class NetgsmSaglayici implements SmsGonderimInterface
{
    /**
     * Netgsm API endpoint'i.
     */
    private const API_URL = 'https://api.netgsm.com.tr/sms/send/xml';

    /**
     * Belirtilen numaralara Netgsm üzerinden SMS gönderir.
     *
     * @param array $ayarlar API ayarları ['username', 'password']
     * @param string $baslik Gönderen başlığı (Originator)
     * @param string $mesaj Gönderilecek mesaj içeriği
     * @param array|string $numaralar Alıcı telefon numaraları
     * @return array Gönderim sonucu
     */
    public function gonder(array $ayarlar, string $baslik, string $mesaj, $numaralar): array
    {
        if (empty($ayarlar['username']) || empty($ayarlar['password'])) {
            return ['status' => 'error', 'message' => 'Netgsm API bilgileri eksik.'];
        }

        $numaralar_str = is_array($numaralar) ? implode('', array_map(fn($no) => "<no>{$no}</no>", $numaralar)) : "<no>{$numaralar}</no>";

        $xml_body = <<<XML
        <?xml version="1.0" encoding="UTF-8"?>
        <mainbody>
            <header>
                <usercode>{$ayarlar['username']}</usercode>
                <password>{$ayarlar['password']}</password>
                <msgheader>{$baslik}</msgheader>
            </header>
            <body>
                <msg><![CDATA[{$mesaj}]]></msg>
                {$numaralar_str}
            </body>
        </mainbody>
        XML;

        try {
            $response = Http::withHeaders([
                'Content-Type' => 'text/xml; charset=UTF-8',
            ])->send('POST', self::API_URL, [
                'body' => $xml_body
            ]);

            if ($response->failed()) {
                return ['status' => 'error', 'message' => 'API isteği başarısız oldu.', 'data' => $response->body()];
            }

            $responseBody = $response->body();
            // Netgsm'den gelen yanıt kodları: 00, 01, 02 başarılı, diğerleri hata.
            // Örnek yanıt: "00 123456789" (başarılı, jobid döndürür) veya "30" (hatalı şifre)
            $resultCode = substr($responseBody, 0, 2);

            if (in_array($resultCode, ['00', '01', '02'])) {
                return ['status' => 'success', 'message' => 'SMS başarıyla gönderim için sıraya alındı.', 'data' => $responseBody];
            } else {
                 return ['status' => 'error', 'message' => $this->getErrorMessage($resultCode), 'data' => $responseBody];
            }

        } catch (\Exception $e) {
            return ['status' => 'error', 'message' => 'SMS gönderimi sırasında bir istisna oluştu: ' . $e->getMessage()];
        }
    }

    /**
     * Netgsm API hata kodunu anlamlı bir mesaja çevirir.
     *
     * @param string $code
     * @return string
     */
    private function getErrorMessage(string $code): string
    {
        $errors = [
            '20' => 'Mesaj metni veya formatı hatalı.',
            '30' => 'Geçersiz kullanıcı adı, şifre veya API kullanıcısı aktif değil.',
            '40' => 'Mesaj başlığı (gönderen adı) tanımlı değil.',
            '41' => 'Arama engeli.',
            '42' => 'Ulaşım engeli.',
            '43' => 'Mükerrer gönderim.',
            '50' => 'Gönderilecek numara hatalı.',
            '51' => 'Gönderim sınırını aştınız.',
            '60' => 'Hesapta yeterli kredi yok.',
            '70' => 'API erişimi IP kısıtlaması.',
            '80' => 'Sorgu gönderim hızı aşıldı.',
            '81' => 'Sistem hatası.',
            '82' => 'İşlem zaman aşımına uğradı.',
            '85' => 'Genel bir hata oluştu.',
        ];

        return $errors[$code] ?? "Bilinmeyen hata kodu: {$code}";
    }
}
