<?php

namespace App\Services\Sms;

use Illuminate\Support\Facades\Http;

/**
 * Class JetSmsSaglayici
 * JetSMS API'si ile SMS gönderimini gerçekleştirir.
 * Dokümantasyon: https://www.jetsms.com.tr/api-dokumantasyonu
 */
class JetSmsSaglayici implements SmsGonderimInterface
{
    /**
     * JetSMS API endpoint'i.
     */
    private const API_URL = 'https://api.jetsms.com.tr/sms/send/get';

    /**
     * Belirtilen numaralara JetSMS üzerinden SMS gönderir.
     *
     * @param array $ayarlar API ayarları ['username', 'password']
     * @param string $baslik Gönderen başlığı (Originator)
     * @param string $mesaj Gönderilecek mesaj içeriği
     * @param array|string $numaralar Alıcı telefon numaraları
     * @return array Gönderim sonucu
     */
    public function gonder(array $ayarlar, string $baslik, string $mesaj, $numaralar): array
    {
        if (empty($ayarlar['username']) || empty($ayarlar['password'])) {
            return ['status' => 'error', 'message' => 'JetSMS API bilgileri eksik.'];
        }

        // Numaraları virgülle ayır
        $numaralar_str = is_array($numaralar) ? implode(',', $numaralar) : $numaralar;

        try {
            $response = Http::get(self::API_URL, [
                'username' => $ayarlar['username'],
                'password' => $ayarlar['password'],
                'header' => $baslik,
                'message' => $mesaj,
                'gsm' => $numaralar_str,
            ]);

            if ($response->failed()) {
                return ['status' => 'error', 'message' => 'API isteği başarısız oldu.', 'data' => $response->body()];
            }

            $responseBody = $response->body();
            
            // JetSMS yanıt formatı: "SUCCESS [JOBID]" veya "ERROR: [HATA MESAJI]"
            if (str_starts_with($responseBody, 'SUCCESS')) {
                $jobId = trim(str_replace('SUCCESS', '', $responseBody));
                return [
                    'status' => 'success', 
                    'message' => 'SMS başarıyla gönderim için sıraya alındı.', 
                    'data' => ['message_id' => $jobId, 'raw' => $responseBody]
                ];
            } else {
                return [
                    'status' => 'error', 
                    'message' => $this->getErrorMessage($responseBody), 
                    'data' => $responseBody
                ];
            }

        } catch (\Exception $e) {
            return ['status' => 'error', 'message' => 'SMS gönderimi sırasında bir istisna oluştu: ' . $e->getMessage()];
        }
    }

    /**
     * JetSMS API hata mesajını anlamlı bir mesaja çevirir.
     *
     * @param string $errorBody
     * @return string
     */
    private function getErrorMessage(string $errorBody): string
    {
        // JetSMS hata formatı genelde "ERROR: açıklama" şeklindedir
        if (str_contains($errorBody, 'ERROR:')) {
            return trim(str_replace('ERROR:', '', $errorBody));
        }

        // Yaygın hata kodları
        $errors = [
            'INVALID_USER_PASSWORD' => 'Geçersiz kullanıcı adı veya şifre',
            'INVALID_NUMBERS' => 'Geçersiz telefon numarası formatı',
            'INSUFFICIENT_CREDITS' => 'Yetersiz SMS kredisi',
            'INVALID_SENDER' => 'Geçersiz gönderici başlığı',
            'BLACKLISTED' => 'Numara kara listede',
        ];

        foreach ($errors as $code => $message) {
            if (str_contains($errorBody, $code)) {
                return $message;
            }
        }

        return 'JetSMS API hatası: ' . $errorBody;
    }
}
