<?php

namespace App\Http\Controllers;

use App\Services\SmsServisi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class KullaniciPanelController extends Controller
{
    /**
     * Kullanıcı dashboard sayfasını gösterir.
     */
    public function dashboard(Request $request)
    {
        $user = Auth::guard('kullanici')->user();
        $kurumId = $request->attributes->get('kurum_id');

        // İstatistikler
        $stats = [
            // Kullanıcının kan grubundaki toplam kullanıcı sayısı
            'ayni_kan_grubu' => DB::table('kullanicilar')
                ->where('kurum_id', $kurumId)
                ->where('kan_grubu_id', $user->kan_grubu_id)
                ->count(),
            
            // Toplam kullanıcı sayısı
            'toplam_kullanici' => DB::table('kullanicilar')
                ->where('kurum_id', $kurumId)
                ->count(),
            
            // Bugün verilen ilan sayısı
            'bugun_ilanlar' => DB::table('kan_ilanlari')
                ->where('kurum_id', $kurumId)
                ->whereDate('created_at', today())
                ->count(),
            
            // Tamamlanan (pasif) ilan sayısı
            'tamamlanan_ilanlar' => DB::table('kan_ilanlari')
                ->where('kurum_id', $kurumId)
                ->where('durum', 'pasif')
                ->count(),
        ];

        // İl listesi - filtreleme için
        $iller = DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = DB::table('kurum_iller')->orderBy('ad')->get();
        }

        // Aktif ilanlar - filtreleme ile (sadece aynı kan grubu)
        $query = DB::table('kan_ilanlari')
            ->leftJoin('kan_gruplari', 'kan_ilanlari.kan_grubu_id', '=', 'kan_gruplari.id')
            ->leftJoin('iller', 'kan_ilanlari.il_id', '=', 'iller.id')
            ->leftJoin('kurum_iller', 'kan_ilanlari.il_id', '=', 'kurum_iller.id')
            ->leftJoin('ilceler', 'kan_ilanlari.ilce_id', '=', 'ilceler.id')
            ->leftJoin('kurum_ilceler', 'kan_ilanlari.ilce_id', '=', 'kurum_ilceler.id')
            ->select('kan_ilanlari.*', 
                'kan_gruplari.ad as kan_grubu',
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_il, ""), COALESCE(iller.ad, kurum_iller.ad)) as il_adi'),
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_ilce, ""), COALESCE(ilceler.ad, kurum_ilceler.ad)) as ilce_adi'))
            ->where('kan_ilanlari.kurum_id', $kurumId)
            ->where('kan_ilanlari.durum', 'aktif')
            ->where('kan_ilanlari.kan_grubu_id', $user->kan_grubu_id);

        // Filtreleme
        if ($request->has('il_id') && $request->il_id) {
            $query->where('kan_ilanlari.il_id', $request->il_id);
        }
        if ($request->has('ilce_id') && $request->ilce_id) {
            $query->where('kan_ilanlari.ilce_id', $request->ilce_id);
        }

        $ilanlar = $query->orderBy('kan_ilanlari.created_at', 'desc')->paginate(10);

        // Seçili il varsa ilçeleri getir
        $ilceler = collect();
        if ($request->has('il_id') && $request->il_id) {
            $ilceler = DB::table('ilceler')->where('il_id', $request->il_id)->orderBy('ad')->get();
            if ($ilceler->isEmpty()) {
                $ilceler = DB::table('kurum_ilceler')->where('kurum_il_id', $request->il_id)->orderBy('ad')->get();
            }
        }

        return view('kullanici.dashboard', compact('ilanlar', 'stats', 'iller', 'ilceler'));
    }

    /**
     * Kullanıcının profil sayfasını gösterir.
     */
    public function profil()
    {
        $user = Auth::guard('kullanici')->user();
        return view('kullanici.profil', compact('user'));
    }

    /**
     * Kullanıcı profil bilgilerini günceller.
     */
    public function updateProfil(Request $request)
    {
        $user = Auth::guard('kullanici')->user();

        // Kan grubu değişikliği var mı kontrol et
        $kanGrubuDegisti = false;
        $eskiKanGrubuId = $user->kan_grubu_id;
        $yeniKanGrubuId = null;
        
        if ($request->has('kan_grubu_id') && $request->kan_grubu_id != $eskiKanGrubuId) {
            $kanGrubuDegisti = true;
            $yeniKanGrubuId = $request->kan_grubu_id;
        }

        $request->validate([
            'ad' => 'required|string|max:255',
            'soyad' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('kullanicilar')->ignore($user->id)],
            'telefon' => ['required', 'string', Rule::unique('kullanicilar')->ignore($user->id)],
            'dogum_tarihi' => 'required|date',
            'kan_grubu_id' => 'nullable|exists:kan_gruplari,id',
            'otp_code' => $kanGrubuDegisti ? 'required|string|size:6' : 'nullable',
            'sms_izni' => 'nullable|boolean',
        ]);

        // Kan grubu değişiyorsa OTP doğrula
        if ($kanGrubuDegisti) {
            $otpKayit = DB::table('oturum_dogrulama_kodlari')
                ->where('telefon', $user->telefon)
                ->where('kod', $request->otp_code)
                ->where('kullanildi_mi', false)
                ->where('son_kullanim_tarihi', '>', now())
                ->first();

            if (!$otpKayit) {
                return back()->withErrors(['otp_code' => 'Geçersiz veya süresi dolmuş OTP kodu.'])->withInput();
            }

            // OTP'yi kullanıldı olarak işaretle
            DB::table('oturum_dogrulama_kodlari')
                ->where('id', $otpKayit->id)
                ->update(['kullanildi_mi' => true]);
        }

        $data = $request->only(['ad', 'soyad', 'email', 'telefon', 'dogum_tarihi']);
        
        // SMS izni
        $data['sms_izni'] = $request->has('sms_izni') ? true : false;
        
        // Kan grubu değişiyorsa ekle
        if ($kanGrubuDegisti) {
            $data['kan_grubu_id'] = $yeniKanGrubuId;
        }

        DB::table('kullanicilar')->where('id', $user->id)->update($data);

        $mesaj = 'Profil bilgileriniz başarıyla güncellendi.';
        if ($kanGrubuDegisti) {
            $mesaj = 'Profil bilgileriniz ve kan grubunuz başarıyla güncellendi.';
        }

        return redirect()->route('kullanici.profil')->with('success', $mesaj);
    }

    /**
     * Yeni kan ilanı oluşturma formunu gösterir.
     */
    public function showIlanForm(Request $request)
    {
        $user = Auth::guard('kullanici')->user();
        
        // Kullanıcının kan grubunu al
        $kanGrubu = 'Bilinmiyor';
        if ($user->kan_grubu_id) {
            $kanGrubu = DB::table('kan_gruplari')
                ->where('id', $user->kan_grubu_id)
                ->value('ad') ?? 'Bilinmiyor';
        }
        
        // İl listesini al - fallback ile
        $iller = DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = DB::table('kurum_iller')->orderBy('ad')->get();
        }
        
        return view('kullanici.ilan-ver', compact('iller', 'kanGrubu', 'user'));
    }

    /**
     * Yeni kan ilanını veritabanına kaydeder.
     */
    public function storeIlan(Request $request)
    {
        $user = Auth::guard('kullanici')->user();
        $kurumId = $request->attributes->get('kurum_id');

        // Kullanıcının kan grubu kontrolü
        if (!$user->kan_grubu_id) {
            return back()->withErrors(['error' => 'İlan verebilmek için önce profilinizden kan grubunuzu güncellemeniz gerekmektedir.'])->withInput();
        }

        $request->validate([
            'unite' => 'required|integer|min:1',
            'hastane' => 'required|string|max:255',
            'hastane_adresi' => 'required|string|max:500',
            'bagis_turu' => 'required|in:kan_unitesi,canli_bagisci',
            'hastane_il' => 'required|string|max:255',
            'hastane_ilce' => 'required|string|max:255',
            'iletisim_telefon' => 'required|string|max:20',
            'yakin_kisi_telefon' => 'nullable|string|max:20',
            'aciklama' => 'nullable|string',
            'otp_code' => 'required|string|size:6',
        ]);

        // SMS onayını kontrol et
        $smsOnay = DB::table('ilan_sms_onaylari')
            ->where('kullanici_id', $user->id)
            ->where('otp_kod', $request->otp_code)
            ->where('onaylandi_mi', false)
            ->where('created_at', '>', now()->subMinutes(5))
            ->first();

        if (!$smsOnay) {
            return back()->withErrors(['otp_code' => 'Geçersiz veya süresi dolmuş OTP kodu.'])->withInput();
        }

        // Benzersiz slug oluştur
        $slug = \Str::random(12) . '-' . time();
        
        // Link token ve geçerlilik tarihi oluştur
        $tokenData = generate_ilan_link_token(0); // ID henüz yok, 0 geçiyoruz
        
        // Kullanıcının kan grubunu ve adını kullan (kullanıcı sadece kendisi için ilan verebilir)
        $ilanId = DB::table('kan_ilanlari')->insertGetId([
            'slug' => $slug,
            'kurum_id' => $kurumId,
            'kullanici_id' => $user->id,
            'hasta_ad_soyad' => $user->ad . ' ' . $user->soyad,
            'hastane_adi' => $request->hastane,
            'hastane_adresi' => $request->hastane_adresi,
            'kan_grubu_id' => $user->kan_grubu_id,
            'unite_sayisi' => $request->unite,
            'bagis_turu' => $request->bagis_turu,
            'il_id' => null,
            'ilce_id' => null,
            'hastane_il' => $request->hastane_il,
            'hastane_ilce' => $request->hastane_ilce,
            'irtibat_telefonu' => $request->iletisim_telefon,
            'yakin_kisi_telefon' => $request->yakin_kisi_telefon,
            'notlar' => $request->aciklama,
            'durum' => 'aktif',
            'link_token' => $tokenData['token'],
            'link_gecerlilik_tarihi' => $tokenData['gecerlilik_tarihi'],
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // SMS onayını güncelle
        DB::table('ilan_sms_onaylari')
            ->where('id', $smsOnay->id)
            ->update([
                'ilan_id' => $ilanId,
                'onaylandi_mi' => true,
                'onay_tarihi' => now(),
                'ip_adresi' => $request->ip(),
            ]);

        // Faz 6: İlan sonrası SMS gönderme işlevi
        $ilan = DB::table('kan_ilanlari')->find($ilanId);
        $ilanLink = url('/ilan/' . $ilan->slug);
        $smsServisi = new SmsServisi();
        $smsSonuc = $smsServisi->ilanSmsGonder($kurumId, $ilan, $ilanLink);

        $flashMessage = 'Kan ilanınız başarıyla oluşturuldu.';
        if ($smsSonuc['status'] === 'success') {
            $flashMessage .= ' ' . $smsSonuc['message'];
        } elseif ($smsSonuc['status'] === 'warning') {
            $flashMessage .= ' Ancak SMS gönderiminde bazı sorunlar yaşandı: ' . implode(', ', $smsSonuc['errors']);
        } elseif ($smsSonuc['status'] === 'error') {
            $flashMessage .= ' Ancak SMS gönderimi başarısız oldu: ' . $smsSonuc['message'];
        }


        return redirect()->route('kullanici.ilanlarim')->with('success', $flashMessage);
    }

    /**
     * Kullanıcının kendi ilanlarını listeler.
     */
    public function ilanlarim(Request $request)
    {
        $user = Auth::guard('kullanici')->user();
        $kurumId = $request->attributes->get('kurum_id');

        // İl listesi
        $iller = DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = DB::table('kurum_iller')->orderBy('ad')->get();
        }

        // İlanlar sorgusu
        $query = DB::table('kan_ilanlari')
            ->leftJoin('kan_gruplari', 'kan_ilanlari.kan_grubu_id', '=', 'kan_gruplari.id')
            ->leftJoin('iller', 'kan_ilanlari.il_id', '=', 'iller.id')
            ->leftJoin('kurum_iller', 'kan_ilanlari.il_id', '=', 'kurum_iller.id')
            ->leftJoin('ilceler', 'kan_ilanlari.ilce_id', '=', 'ilceler.id')
            ->leftJoin('kurum_ilceler', 'kan_ilanlari.ilce_id', '=', 'kurum_ilceler.id')
            ->select('kan_ilanlari.*', 
                'kan_gruplari.ad as kan_grubu',
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_il, ""), COALESCE(iller.ad, kurum_iller.ad)) as il_adi'),
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_ilce, ""), COALESCE(ilceler.ad, kurum_ilceler.ad)) as ilce_adi'))
            ->where('kan_ilanlari.kurum_id', $kurumId)
            ->where('kan_ilanlari.kullanici_id', $user->id);

        // Filtreleme
        if ($request->has('il_id') && $request->il_id) {
            $query->where('kan_ilanlari.il_id', $request->il_id);
        }
        if ($request->has('ilce_id') && $request->ilce_id) {
            $query->where('kan_ilanlari.ilce_id', $request->ilce_id);
        }

        $ilanlar = $query->orderBy('kan_ilanlari.created_at', 'desc')->paginate(10);

        // Seçili il varsa ilçeleri getir
        $ilceler = collect();
        if ($request->has('il_id') && $request->il_id) {
            $ilceler = DB::table('ilceler')->where('il_id', $request->il_id)->orderBy('ad')->get();
            if ($ilceler->isEmpty()) {
                $ilceler = DB::table('kurum_ilceler')->where('kurum_il_id', $request->il_id)->orderBy('ad')->get();
            }
        }

        return view('kullanici.ilanlarim', compact('ilanlar', 'iller', 'ilceler'));
    }

    /**
     * İlan detayını gösterir (kullanıcı paneli içinde).
     */
    public function showIlan($id)
    {
        $user = Auth::guard('kullanici')->user();
        
        // İlanı kullanıcıya ait olup olmadığını kontrol ederek getir
        $ilan = DB::table('kan_ilanlari')
            ->leftJoin('kan_gruplari', 'kan_ilanlari.kan_grubu_id', '=', 'kan_gruplari.id')
            ->leftJoin('kullanicilar', 'kan_ilanlari.kullanici_id', '=', 'kullanicilar.id')
            ->leftJoin('iller', 'kan_ilanlari.il_id', '=', 'iller.id')
            ->leftJoin('kurum_iller', 'kan_ilanlari.il_id', '=', 'kurum_iller.id')
            ->leftJoin('ilceler', 'kan_ilanlari.ilce_id', '=', 'ilceler.id')
            ->leftJoin('kurum_ilceler', 'kan_ilanlari.ilce_id', '=', 'kurum_ilceler.id')
            ->leftJoin('kurumlar', 'kan_ilanlari.kurum_id', '=', 'kurumlar.id')
            ->select(
                'kan_ilanlari.*',
                'kan_gruplari.ad as kan_grubu',
                'kullanicilar.ad as kullanici_ad',
                'kullanicilar.soyad as kullanici_soyad',
                'kullanicilar.telefon as kullanici_telefon',
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_il, ""), COALESCE(iller.ad, kurum_iller.ad)) as il_adi'),
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_ilce, ""), COALESCE(ilceler.ad, kurum_ilceler.ad)) as ilce_adi'),
                'kurumlar.ad as kurum_adi'
            )
            ->where('kan_ilanlari.id', $id)
            ->where('kan_ilanlari.kullanici_id', $user->id)
            ->first();

        if (!$ilan) {
            return redirect()->route('kullanici.ilanlarim')->with('error', 'İlan bulunamadı veya erişim yetkiniz yok.');
        }

        // İl listesi
        $iller = DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = DB::table('kurum_iller')->orderBy('ad')->get();
        }

        // İlçe listesi
        $ilceler = collect();
        if ($ilan->il_id) {
            $ilceler = DB::table('ilceler')->where('il_id', $ilan->il_id)->orderBy('ad')->get();
            if ($ilceler->isEmpty()) {
                $ilceler = DB::table('kurum_ilceler')->where('kurum_il_id', $ilan->il_id)->orderBy('ad')->get();
            }
        }

        // Kan grupları
        return view('kullanici.ilan-show', compact('ilan'));
    }

    /**
     * İlan düzenleme formunu gösterir.
     */
    public function editIlan($id)
    {
        $user = Auth::guard('kullanici')->user();
        
        // İlanı kullanıcıya ait olup olmadığını kontrol ederek getir
        $ilan = DB::table('kan_ilanlari')
            ->leftJoin('kan_gruplari', 'kan_ilanlari.kan_grubu_id', '=', 'kan_gruplari.id')
            ->leftJoin('kullanicilar', 'kan_ilanlari.kullanici_id', '=', 'kullanicilar.id')
            ->leftJoin('iller', 'kan_ilanlari.il_id', '=', 'iller.id')
            ->leftJoin('kurum_iller', 'kan_ilanlari.il_id', '=', 'kurum_iller.id')
            ->leftJoin('ilceler', 'kan_ilanlari.ilce_id', '=', 'ilceler.id')
            ->leftJoin('kurum_ilceler', 'kan_ilanlari.ilce_id', '=', 'kurum_ilceler.id')
            ->select(
                'kan_ilanlari.*',
                'kan_gruplari.ad as kan_grubu',
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_il, ""), COALESCE(iller.ad, kurum_iller.ad)) as il_adi'),
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_ilce, ""), COALESCE(ilceler.ad, kurum_ilceler.ad)) as ilce_adi')
            )
            ->where('kan_ilanlari.id', $id)
            ->where('kan_ilanlari.kullanici_id', $user->id)
            ->first();

        if (!$ilan) {
            return redirect()->route('kullanici.ilanlarim')->with('error', 'İlan bulunamadı veya erişim yetkiniz yok.');
        }

        // İl listesi
        $iller = DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = DB::table('kurum_iller')->orderBy('ad')->get();
        }

        // İlçe listesi
        $ilceler = collect();
        if ($ilan->il_id) {
            $ilceler = DB::table('ilceler')->where('il_id', $ilan->il_id)->orderBy('ad')->get();
            if ($ilceler->isEmpty()) {
                $ilceler = DB::table('kurum_ilceler')->where('kurum_il_id', $ilan->il_id)->orderBy('ad')->get();
            }
        }

        // Kan grupları
        $kanGruplari = DB::table('kan_gruplari')->orderBy('ad')->get();

        return view('kullanici.ilan-edit', compact('ilan', 'iller', 'ilceler', 'kanGruplari'));
    }

    /**
     * İlanı günceller.
     */
    public function updateIlan(Request $request, $id)
    {
        $user = Auth::guard('kullanici')->user();
        
        // İlanın kullanıcıya ait olduğunu kontrol et
        $ilan = DB::table('kan_ilanlari')
            ->where('id', $id)
            ->where('kullanici_id', $user->id)
            ->first();

        if (!$ilan) {
            return redirect()->route('kullanici.ilanlarim')->with('error', 'İlan bulunamadı veya erişim yetkiniz yok.');
        }

        $validated = $request->validate([
            'hasta_ad_soyad' => 'required|string|max:255',
            'kan_grubu_id' => 'required|exists:kan_gruplari,id',
            'hastane' => 'required|string|max:255',
            'hastane_adresi' => 'required|string|max:500',
            'unite' => 'required|integer|min:1',
            'bagis_turu' => 'required|in:kan_unitesi,canli_bagisci',
            'il_id' => 'required|integer',
            'ilce_id' => 'nullable|integer',
            'irtibat_telefonu' => 'required|string|max:20',
            'yakin_kisi_telefon' => 'nullable|string|max:20',
            'aciklama' => 'nullable|string',
        ]);

        DB::table('kan_ilanlari')
            ->where('id', $id)
            ->update([
                'hasta_ad_soyad' => $validated['hasta_ad_soyad'],
                'kan_grubu_id' => $validated['kan_grubu_id'],
                'hastane_adi' => $validated['hastane'],
                'hastane_adresi' => $validated['hastane_adresi'],
                'unite_sayisi' => $validated['unite'],
                'bagis_turu' => $validated['bagis_turu'],
                'il_id' => $validated['il_id'],
                'ilce_id' => $validated['ilce_id'],
                'irtibat_telefonu' => $validated['irtibat_telefonu'],
                'yakin_kisi_telefon' => $validated['yakin_kisi_telefon'],
                'notlar' => $validated['aciklama'],
                'updated_at' => now(),
            ]);

        return redirect()->route('kullanici.ilan.show', $id)->with('success', 'İlan başarıyla güncellendi.');
    }

    /**
     * İlan durumunu günceller (aktif/pasif).
     */
    public function updateIlanStatus(Request $request, $id)
    {
        $user = Auth::guard('kullanici')->user();
        
        // İlanın kullanıcıya ait olduğunu kontrol et
        $ilan = DB::table('kan_ilanlari')
            ->where('id', $id)
            ->where('kullanici_id', $user->id)
            ->first();

        if (!$ilan) {
            return response()->json(['success' => false, 'message' => 'İlan bulunamadı.'], 404);
        }

        $validated = $request->validate([
            'durum' => 'required|in:aktif,pasif,tamamlandi,iptal',
        ]);

        DB::table('kan_ilanlari')
            ->where('id', $id)
            ->update([
                'durum' => $validated['durum'],
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true, 
            'message' => 'İlan durumu güncellendi.',
            'durum' => $validated['durum']
        ]);
    }

    /**
     * Kan grubu değişimi için OTP gönderir.
     */
    public function sendKanGrubuOtp(Request $request)
    {
        $user = Auth::guard('kullanici')->user();
        
        // 6 haneli rastgele kod oluştur
        $kod = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
        
        // Veritabanına kaydet
        DB::table('oturum_dogrulama_kodlari')->insert([
            'telefon' => $user->telefon,
            'kod' => $kod,
            'son_kullanim_tarihi' => now()->addMinutes(5),
            'kullanildi_mi' => false,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // OTP kodunu log dosyasına yaz (test amaçlı)
        \Log::info('Kan Grubu Değişimi OTP', [
            'kullanici_id' => $user->id,
            'telefon' => $user->telefon,
            'otp_kod' => $kod,
            'gecerlilik' => now()->addMinutes(5)->toDateTimeString()
        ]);
        
        // SMS gönder (gerçek SMS servisi varsa)
        // SmsServisi ile gönderilebilir
        
        return response()->json([
            'success' => true,
            'message' => 'OTP kodu telefonunuza gönderildi. Lütfen log dosyasını kontrol edin.',
        ]);
    }

    /**
     * İlan vermek için OTP gönderir.
     */
    public function sendIlanOtp(Request $request)
    {
        \Log::info('sendIlanOtp metodu çağrıldı');
        
        $user = Auth::guard('kullanici')->user();
        
        \Log::info('Kullanıcı bilgisi alındı', ['user_id' => $user->id, 'telefon' => $user->telefon]);
        
        // 6 haneli rastgele kod oluştur
        $kod = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
        
        \Log::info('OTP kodu oluşturuldu', ['kod' => $kod]);
        
        // SMS onayını kaydet
        DB::table('ilan_sms_onaylari')->insert([
            'kullanici_id' => $user->id,
            'otp_kod' => $kod,
            'onaylandi_mi' => false,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // OTP kodunu log dosyasına yaz (test amaçlı)
        \Log::info('İlan Verme OTP', [
            'kullanici_id' => $user->id,
            'telefon' => $user->telefon,
            'otp_kod' => $kod,
            'gecerlilik' => now()->addMinutes(5)->toDateTimeString()
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'OTP kodu telefonunuza gönderildi.',
            'otp_code' => $kod // Test için - SMS API entegre edilince kaldırılacak
        ]);
    }

    /**
     * İlan detay sayfasını gösterir.
     */
    public function ilanDetay($slug)
    {
        $ilan = DB::table('kan_ilanlari')
            ->leftJoin('kan_gruplari', 'kan_ilanlari.kan_grubu_id', '=', 'kan_gruplari.id')
            ->leftJoin('kullanicilar', 'kan_ilanlari.kullanici_id', '=', 'kullanicilar.id')
            ->leftJoin('iller', 'kan_ilanlari.il_id', '=', 'iller.id')
            ->leftJoin('kurum_iller', 'kan_ilanlari.il_id', '=', 'kurum_iller.id')
            ->leftJoin('ilceler', 'kan_ilanlari.ilce_id', '=', 'ilceler.id')
            ->leftJoin('kurum_ilceler', 'kan_ilanlari.ilce_id', '=', 'kurum_ilceler.id')
            ->select(
                'kan_ilanlari.*',
                'kan_gruplari.ad as kan_grubu_adi',
                DB::raw('CONCAT(kullanicilar.ad, " ", kullanicilar.soyad) as kullanici_adi'),
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_il, ""), COALESCE(iller.ad, kurum_iller.ad)) as il_adi'),
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_ilce, ""), COALESCE(ilceler.ad, kurum_ilceler.ad)) as ilce_adi')
            )
            ->where('kan_ilanlari.slug', $slug)
            ->first();

        if (!$ilan) {
            abort(404, 'İlan bulunamadı.');
        }

        // İlan linklerini oluştur
        $ilanLink = url('/ilan/' . $ilan->slug);
        
        // SMS linki (token bazlı, 72 saat geçerli, 5 karakter)
        $smsLink = null;
        if ($ilan->link_token) {
            $smsLink = route('ilan.link', ['token' => $ilan->link_token]);
        }

        // SMS gönderim sayısını al
        $smsSayisi = DB::table('kan_ilan_smsleri')
            ->where('kan_ilan_id', $ilan->id)
            ->where('sms_durumu', 'sent')
            ->count();

        return view('kullanici.ilan-detay', compact('ilan', 'ilanLink', 'smsLink', 'smsSayisi'));
    }
}
