<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Kullanici;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;

class KullaniciAuthController extends Controller
{
    public function showRegistrationForm(Request $request)
    {
        $kanGruplari = \DB::table('kan_gruplari')->get();
        // Eğer iller tablosu boşsa kurum_iller'den çek
        $iller = \DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = \DB::table('kurum_iller')->orderBy('ad')->get();
        }
        
        $kurumAdi = $request->attributes->get('kurum_adi') ?? 'Kan Bankası Sistemi';
        
        return view('auth.register', compact('kanGruplari', 'iller', 'kurumAdi'));
    }

    public function register(Request $request)
    {
        // İl ve ilçe için hangi tabloyu kullanacağımızı belirle
        $illerTable = \DB::table('iller')->count() > 0 ? 'iller' : 'kurum_iller';
        $ilcelerTable = \DB::table('ilceler')->count() > 0 ? 'ilceler' : 'kurum_ilceler';
        
        $request->validate([
            'ad' => ['required', 'string', 'max:255'],
            'soyad' => ['required', 'string', 'max:255'],
            'telefon' => ['required', 'string', 'size:10', 'unique:kullanicilar', 'regex:/^5[0-9]{9}$/'],
            'tc_kimlik' => ['required', 'string', 'size:11', 'unique:kullanicilar', 'regex:/^[1-9][0-9]{10}$/'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:kullanicilar'],
            'dogum_tarihi' => ['required', 'date'],
            'kan_grubu_id' => ['required', 'exists:kan_gruplari,id'],
            'il_id' => ['required', 'exists:' . $illerTable . ',id'],
            'ilce_id' => ['required', 'exists:' . $ilcelerTable . ',id'],
            'belde' => ['nullable', 'string', 'max:100'],
            'mahalle' => ['required', 'string', 'max:100'],
            'kullanim_kosullari' => ['required', 'accepted'],
        ]);

        $kurumId = $request->attributes->get('kurum_id');

        if (!$kurumId) {
            // Eğer bir şekilde kurum çözümlenemediyse, hata ver.
            // Bu normalde olmamalı çünkü middleware'den geçiyor.
            return back()->withErrors(['kurum' => 'Sistemde bir hata oluştu, lütfen daha sonra tekrar deneyin.'])->withInput();
        }

        $user = Kullanici::create([
            'kurum_id' => $kurumId,
            'ad' => $request->ad,
            'soyad' => $request->soyad,
            'kullanici_adi' => $request->telefon,
            'email' => $request->email,
            'telefon' => $request->telefon,
            'tc_kimlik' => $request->tc_kimlik,
            'dogum_tarihi' => $request->dogum_tarihi,
            'kan_grubu_id' => $request->kan_grubu_id,
            'il_id' => $request->il_id,
            'ilce_id' => $request->ilce_id,
            'belde' => $request->belde,
            'mahalle' => $request->mahalle,
            'rol' => 'kullanici',
            'sms_izni' => true,
        ]);

        return redirect()->route('giris')->with('success', 'Kayıt başarılı! Giriş yapmak için telefon ve TC numaranızı kullanın.');
    }

    public function showLoginForm(Request $request)
    {
        $kurumAdi = $request->attributes->get('kurum_adi') ?? 'Kan Bankası Sistemi';
        return view('auth.login', compact('kurumAdi'));
    }

    public function sendOtp(Request $request)
    {
        $request->validate([
            'telefon' => ['required', 'string', 'size:10'],
            'tc_kimlik' => ['required', 'string', 'size:11'],
        ]);

        $kurumId = $request->attributes->get('kurum_id') ?? 1;
        
        $user = Kullanici::where('telefon', $request->telefon)
            ->where('tc_kimlik', $request->tc_kimlik)
            ->where('kurum_id', $kurumId)
            ->first();

        if (!$user) {
            return response()->json(['error' => 'Kullanıcı bulunamadı'], 404);
        }

        // OTP kodu oluştur (6 haneli)
        $otpCode = rand(100000, 999999);
        
        // OTP'yi veritabanına kaydet
        \DB::table('oturum_dogrulama_kodlari')->updateOrInsert(
            ['telefon' => $request->telefon],
            [
                'kod' => $otpCode,
                'son_kullanim_tarihi' => now()->addMinutes(5),
                'kullanildi_mi' => false,
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'created_at' => now(),
                'updated_at' => now(),
            ]
        );

        // Kurum SMS ayarlarını al
        $smsAyarlari = \DB::table('kurum_sms_ayarlari')->where('kurum_id', $kurumId)->first();
        
        if ($smsAyarlari) {
            // SMS gönder
            try {
                $apiAyarlari = [
                    'username' => $smsAyarlari->api_key,
                    'password' => $smsAyarlari->api_secret,
                ];
                $baslik = $smsAyarlari->api_header;
                $mesaj = "Giriş kodunuz: {$otpCode}\nBu kod 5 dakika geçerlidir.";
                
                // Seçilen operatöre göre dinamik provider
                $smsProvider = \App\Services\Sms\SmsFactory::create($smsAyarlari->servis_saglayici ?? 'netgsm');
                $sonuc = $smsProvider->gonder($apiAyarlari, $baslik, $mesaj, [$request->telefon]);
                
                // SMS gönderimini logla
                \DB::table('sms_kayitlari')->insert([
                    'kurum_id' => $kurumId,
                    'tip' => 'OTP Girişi',
                    'telefon' => $request->telefon,
                    'mesaj' => $mesaj,
                    'hedef_numara_sayisi' => 1,
                    'durum' => $sonuc['status'],
                    'sms_saglayici_mesaj_id' => $sonuc['data']['message_id'] ?? null,
                    'hata_mesaji' => $sonuc['status'] === 'error' ? $sonuc['message'] : null,
                    'olusturulma_tarihi' => now(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
                
                if ($sonuc['status'] !== 'success') {
                    \Log::warning('OTP SMS gönderilemedi: ' . $sonuc['message']);
                }
            } catch (\Exception $e) {
                \Log::error('OTP SMS gönderim hatası: ' . $e->getMessage());
            }
        }
        
        // Log'a da yaz (debug için)
        \Log::info('OTP Kodu: ' . $otpCode . ' - Telefon: ' . $request->telefon);

        return response()->json([
            'success' => true, 
            'message' => 'OTP kodu telefonunuza gönderildi',
        ]);
    }

    public function login(Request $request)
    {
        $request->validate([
            'telefon' => ['required', 'string', 'size:10'],
            'tc_kimlik' => ['required', 'string', 'size:11'],
            'otp_code' => ['required', 'string', 'size:6'],
        ]);

        $kurumId = $request->attributes->get('kurum_id') ?? 1;
        
        $user = Kullanici::where('telefon', $request->telefon)
            ->where('tc_kimlik', $request->tc_kimlik)
            ->where('kurum_id', $kurumId)
            ->first();

        if (!$user) {
            return back()->withErrors(['telefon' => 'Kullanıcı bulunamadı.']);
        }

        // OTP doğrulama
        $otp = \DB::table('oturum_dogrulama_kodlari')
            ->where('telefon', $request->telefon)
            ->where('kod', $request->otp_code)
            ->where('kullanildi_mi', false)
            ->where('son_kullanim_tarihi', '>', now())
            ->first();

        if (!$otp) {
            return back()->withErrors(['otp_code' => 'Geçersiz veya süresi dolmuş OTP kodu.']);
        }

        // OTP'yi kullanıldı olarak işaretle
        \DB::table('oturum_dogrulama_kodlari')
            ->where('id', $otp->id)
            ->update(['kullanildi_mi' => true]);

        Auth::guard('kullanici')->login($user);
        $request->session()->regenerate();

        return redirect()->intended(route('kullanici.dashboard'));
    }

    public function logout(Request $request)
    {
        Auth::guard('kullanici')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect()->route('kullanici.login');
    }
}
