<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

class KullaniciYonetimiController extends Controller
{
    public function index(Request $request)
    {
        $kurumId = $request->attributes->get('kurum_id');
        
        $query = DB::table('kullanicilar')
            ->leftJoin('kan_gruplari', 'kullanicilar.kan_grubu_id', '=', 'kan_gruplari.id')
            ->leftJoin('iller', 'kullanicilar.il_id', '=', 'iller.id')
            ->where('kullanicilar.kurum_id', $kurumId)
            ->select(
                'kullanicilar.*',
                'kan_gruplari.ad as kan_grubu',
                'iller.ad as il_adi'
            );

        // Filtreleme
        if ($request->filled('kan_grubu_id')) {
            $query->where('kullanicilar.kan_grubu_id', $request->kan_grubu_id);
        }

        if ($request->filled('il_id')) {
            $query->where('kullanicilar.il_id', $request->il_id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('kullanicilar.ad', 'like', "%{$search}%")
                  ->orWhere('kullanicilar.soyad', 'like', "%{$search}%")
                  ->orWhere('kullanicilar.email', 'like', "%{$search}%")
                  ->orWhere('kullanicilar.tc_kimlik_no', 'like', "%{$search}%");
            });
        }

        $kullanicilar = $query->orderBy('kullanicilar.created_at', 'desc')->paginate(15);

        // Filtre için dropdown verileri
        $kanGruplari = DB::table('kan_gruplari')->get();
        $iller = DB::table('iller')->orderBy('ad')->get();

        return view('admin.kullanicilar.index', compact('kullanicilar', 'kanGruplari', 'iller'));
    }

    public function create(Request $request)
    {
        $kurumId = $request->attributes->get('kurum_id');
        $kanGruplari = DB::table('kan_gruplari')->get();
        
        // İller - fallback ile
        $iller = DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = DB::table('kurum_iller')->where('kurum_id', $kurumId)->orderBy('ad')->get();
        }

        return view('admin.kullanicilar.create', compact('kanGruplari', 'iller', 'kurumId'));
    }

    public function store(Request $request)
    {
        $kurumId = $request->attributes->get('kurum_id');

        $validated = $request->validate([
            'ad' => ['required', 'string', 'max:255'],
            'soyad' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:kullanicilar,email'],
            'tc_kimlik' => ['nullable', 'string', 'size:11', 'unique:kullanicilar,tc_kimlik'],
            'telefon' => ['required', 'string', 'max:20'],
            'kan_grubu_id' => ['required', 'exists:kan_gruplari,id'],
            'il_id' => ['required', 'integer'],
            'ilce_id' => ['nullable', 'integer'],
            'dogum_tarihi' => ['nullable', 'date'],
            'cinsiyet' => ['nullable', 'in:E,K'],
            'password' => ['nullable', 'confirmed', Password::defaults()],
        ], [
            'ad.required' => 'Ad alanı zorunludur.',
            'soyad.required' => 'Soyad alanı zorunludur.',
            'email.required' => 'E-posta adresi zorunludur.',
            'email.unique' => 'Bu e-posta adresi zaten kullanılıyor.',
            'tc_kimlik.size' => 'TC Kimlik No 11 karakter olmalıdır.',
            'tc_kimlik.unique' => 'Bu TC Kimlik No zaten kayıtlı.',
            'telefon.required' => 'Telefon numarası zorunludur.',
            'kan_grubu_id.required' => 'Kan grubu seçimi zorunludur.',
            'il_id.required' => 'İl seçimi zorunludur.',
            'password.confirmed' => 'Şifre tekrarı eşleşmiyor.',
        ]);
        
        // İl ve ilçe ID'lerinin varlığını manuel kontrol et (iller veya kurum_iller)
        $ilExists = DB::table('iller')->where('id', $validated['il_id'])->exists() || 
                    DB::table('kurum_iller')->where('id', $validated['il_id'])->exists();
        
        if (!$ilExists) {
            return back()->withErrors(['il_id' => 'Seçilen il geçersiz.'])->withInput();
        }
        
        if (isset($validated['ilce_id'])) {
            $ilceExists = DB::table('ilceler')->where('id', $validated['ilce_id'])->exists() || 
                         DB::table('kurum_ilceler')->where('id', $validated['ilce_id'])->exists();
            
            if (!$ilceExists) {
                return back()->withErrors(['ilce_id' => 'Seçilen ilçe geçersiz.'])->withInput();
            }
        }

        $insertData = [
            'kurum_id' => $kurumId,
            'ad' => $validated['ad'],
            'soyad' => $validated['soyad'],
            'email' => $validated['email'],
            'telefon' => $validated['telefon'],
            'kan_grubu_id' => $validated['kan_grubu_id'],
            'il_id' => $validated['il_id'],
            'ilce_id' => $validated['ilce_id'] ?? null,
            'password' => Hash::make($validated['password'] ?? '123456'),
            'aktif' => true,
            'created_at' => now(),
            'updated_at' => now(),
        ];

        if (isset($validated['tc_kimlik'])) {
            $insertData['tc_kimlik'] = $validated['tc_kimlik'];
        }
        if (isset($validated['dogum_tarihi'])) {
            $insertData['dogum_tarihi'] = $validated['dogum_tarihi'];
        }
        if (isset($validated['cinsiyet'])) {
            $insertData['cinsiyet'] = $validated['cinsiyet'];
        }

        DB::table('kullanicilar')->insert($insertData);

        return redirect()->route('admin.kullanicilar.index')->with('success', 'Kullanıcı başarıyla eklendi.');
    }

    public function show(Request $request, $id)
    {
        $kurumId = $request->attributes->get('kurum_id');
        
        $kullanici = DB::table('kullanicilar')
            ->leftJoin('kan_gruplari', 'kullanicilar.kan_grubu_id', '=', 'kan_gruplari.id')
            ->leftJoin('iller', 'kullanicilar.il_id', '=', 'iller.id')
            ->leftJoin('kurum_iller', 'kullanicilar.il_id', '=', 'kurum_iller.id')
            ->leftJoin('ilceler', 'kullanicilar.ilce_id', '=', 'ilceler.id')
            ->leftJoin('kurum_ilceler', 'kullanicilar.ilce_id', '=', 'kurum_ilceler.id')
            ->where('kullanicilar.id', $id)
            ->where('kullanicilar.kurum_id', $kurumId)
            ->select(
                'kullanicilar.*',
                'kan_gruplari.ad as kan_grubu',
                DB::raw('COALESCE(iller.ad, kurum_iller.ad) as il_adi'),
                DB::raw('COALESCE(ilceler.ad, kurum_ilceler.ad) as ilce_adi')
            )
            ->first();

        if (!$kullanici) {
            abort(404);
        }

        // İstatistikler
        $ilanSayisi = DB::table('kan_ilanlari')->where('kullanici_id', $id)->count();
        
        return view('admin.kullanicilar.show', compact('kullanici', 'ilanSayisi'));
    }

    public function edit(Request $request, $id)
    {
        $kurumId = $request->attributes->get('kurum_id');
        
        $kullanici = DB::table('kullanicilar')
            ->where('id', $id)
            ->where('kurum_id', $kurumId)
            ->first();

        if (!$kullanici) {
            abort(404);
        }

        $kanGruplari = DB::table('kan_gruplari')->get();
        
        // İller - fallback ile
        $iller = DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = DB::table('kurum_iller')->where('kurum_id', $kurumId)->orderBy('ad')->get();
        }
        
        // İlçeler - fallback ile
        $ilceler = DB::table('ilceler')->where('il_id', $kullanici->il_id)->orderBy('ad')->get();
        if ($ilceler->isEmpty()) {
            $ilceler = DB::table('kurum_ilceler')->where('kurum_il_id', $kullanici->il_id)->orderBy('ad')->get();
        }

        return view('admin.kullanicilar.edit', compact('kullanici', 'kanGruplari', 'iller', 'ilceler'));
    }

    public function update(Request $request, $id)
    {
        $kurumId = $request->attributes->get('kurum_id');

        $kullanici = DB::table('kullanicilar')
            ->where('id', $id)
            ->where('kurum_id', $kurumId)
            ->first();

        if (!$kullanici) {
            abort(404);
        }

        $validated = $request->validate([
            'ad' => ['required', 'string', 'max:255'],
            'soyad' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:kullanicilar,email,' . $id],
            'tc_kimlik' => ['nullable', 'string', 'size:11', 'unique:kullanicilar,tc_kimlik,' . $id],
            'telefon' => ['required', 'string', 'max:20'],
            'kan_grubu_id' => ['required', 'exists:kan_gruplari,id'],
            'il_id' => ['required', 'integer'],
            'ilce_id' => ['nullable', 'integer'],
            'dogum_tarihi' => ['nullable', 'date'],
            'cinsiyet' => ['nullable', 'in:E,K'],
            'password' => ['nullable', 'confirmed', Password::defaults()],
            'aktif' => ['boolean'],
            'sms_izni' => ['boolean'],
        ]);
        
        // İl ve ilçe ID'lerinin varlığını manuel kontrol et (iller veya kurum_iller)
        $ilExists = DB::table('iller')->where('id', $validated['il_id'])->exists() || 
                    DB::table('kurum_iller')->where('id', $validated['il_id'])->exists();
        
        if (!$ilExists) {
            return back()->withErrors(['il_id' => 'Seçilen il geçersiz.'])->withInput();
        }
        
        if ($validated['ilce_id']) {
            $ilceExists = DB::table('ilceler')->where('id', $validated['ilce_id'])->exists() || 
                         DB::table('kurum_ilceler')->where('id', $validated['ilce_id'])->exists();
            
            if (!$ilceExists) {
                return back()->withErrors(['ilce_id' => 'Seçilen ilçe geçersiz.'])->withInput();
            }
        }

        $updateData = [
            'ad' => $validated['ad'],
            'soyad' => $validated['soyad'],
            'email' => $validated['email'],
            'tc_kimlik' => $validated['tc_kimlik'],
            'telefon' => $validated['telefon'],
            'kan_grubu_id' => $validated['kan_grubu_id'],
            'il_id' => $validated['il_id'],
            'ilce_id' => $validated['ilce_id'] ?? null,
            'aktif' => $request->has('aktif'),
            'sms_izni' => $request->has('sms_izni'),
            'updated_at' => now(),
        ];

        // Sadece varsa ekle
        if (isset($validated['dogum_tarihi'])) {
            $updateData['dogum_tarihi'] = $validated['dogum_tarihi'];
        }
        if (isset($validated['cinsiyet'])) {
            $updateData['cinsiyet'] = $validated['cinsiyet'];
        }

        if ($request->filled('password')) {
            $updateData['password'] = Hash::make($validated['password']);
        }

        DB::table('kullanicilar')->where('id', $id)->update($updateData);

        return redirect()->route('admin.kullanicilar.index')->with('success', 'Kullanıcı başarıyla güncellendi.');
    }

    public function destroy(Request $request, $id)
    {
        $kurumId = $request->attributes->get('kurum_id');

        $kullanici = DB::table('kullanicilar')
            ->where('id', $id)
            ->where('kurum_id', $kurumId)
            ->first();

        if (!$kullanici) {
            abort(404);
        }

        DB::table('kullanicilar')->where('id', $id)->delete();

        return redirect()->route('admin.kullanicilar.index')->with('success', 'Kullanıcı başarıyla silindi.');
    }

    // İlçeleri AJAX ile getir
    public function getIlceler($ilId)
    {
        $ilceler = DB::table('ilceler')
            ->where('il_id', $ilId)
            ->orderBy('ad')
            ->get();
        
        // İlçe bulunamazsa kurum ilçelerinden dene
        if ($ilceler->isEmpty()) {
            $ilceler = DB::table('kurum_ilceler')
                ->where('kurum_il_id', $ilId)
                ->orderBy('ad')
                ->get();
        }

        return response()->json($ilceler);
    }
    
    // Kullanıcının ilanlarını listele
    public function ilanlar(Request $request, $id)
    {
        $kurumId = $request->attributes->get('kurum_id');
        
        $kullanici = DB::table('kullanicilar')
            ->where('id', $id)
            ->where('kurum_id', $kurumId)
            ->first();

        if (!$kullanici) {
            abort(404);
        }

        $ilanlar = DB::table('kan_ilanlari')
            ->leftJoin('kan_gruplari', 'kan_ilanlari.kan_grubu_id', '=', 'kan_gruplari.id')
            ->where('kan_ilanlari.kullanici_id', $id)
            ->where('kan_ilanlari.kurum_id', $kurumId)
            ->select('kan_ilanlari.*', 'kan_gruplari.ad as kan_grubu')
            ->orderBy('kan_ilanlari.created_at', 'desc')
            ->paginate(15);

        return view('admin.kullanicilar.ilanlar', compact('kullanici', 'ilanlar'));
    }
}
