<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\SmsServisi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class KanIlaniYonetimiController extends Controller
{
    public function index(Request $request)
    {
        $kurumId = $request->attributes->get('kurum_id');
        $ilanlar = DB::table('kan_ilanlari')
            ->leftJoin('kan_gruplari', 'kan_ilanlari.kan_grubu_id', '=', 'kan_gruplari.id')
            ->select('kan_ilanlari.*', 'kan_gruplari.ad as kan_grubu')
            ->where('kan_ilanlari.kurum_id', $kurumId)
            ->orderBy('kan_ilanlari.created_at', 'desc')
            ->paginate(15);

        return view('admin.kan-ilanlari.index', compact('ilanlar'));
    }

    public function show(Request $request, $id)
    {
        $kurumId = $request->attributes->get('kurum_id');
        
        $ilan = DB::table('kan_ilanlari')
            ->leftJoin('kan_gruplari', 'kan_ilanlari.kan_grubu_id', '=', 'kan_gruplari.id')
            ->leftJoin('kullanicilar', 'kan_ilanlari.kullanici_id', '=', 'kullanicilar.id')
            ->leftJoin('iller', 'kan_ilanlari.il_id', '=', 'iller.id')
            ->leftJoin('kurum_iller', 'kan_ilanlari.il_id', '=', 'kurum_iller.id')
            ->leftJoin('ilceler', 'kan_ilanlari.ilce_id', '=', 'ilceler.id')
            ->leftJoin('kurum_ilceler', 'kan_ilanlari.ilce_id', '=', 'kurum_ilceler.id')
            ->select(
                'kan_ilanlari.*',
                'kan_gruplari.ad as kan_grubu',
                DB::raw('CONCAT(kullanicilar.ad, " ", kullanicilar.soyad) as kullanici_adi'),
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_il, ""), COALESCE(iller.ad, kurum_iller.ad)) as il_adi'),
                DB::raw('COALESCE(NULLIF(kan_ilanlari.hastane_ilce, ""), COALESCE(ilceler.ad, kurum_ilceler.ad)) as ilce_adi')
            )
            ->where('kan_ilanlari.id', $id)
            ->where('kan_ilanlari.kurum_id', $kurumId)
            ->first();

        if (!$ilan) {
            abort(404, 'İlan bulunamadı.');
        }

        // İlan linkini oluştur (yeni 5 karakterlik token ile)
        $ilanLink = $ilan->link_token ? route('ilan.link', ['token' => $ilan->link_token]) : 'Token oluşturulmamış';

        // SMS gönderim sayısını al
        $smsSayisi = DB::table('kan_ilan_smsleri')
            ->where('kan_ilan_id', $id)
            ->where('sms_durumu', 'sent')
            ->count();

        return view('admin.kan-ilanlari.show', compact('ilan', 'ilanLink', 'smsSayisi'));
    }

    public function updateStatus(Request $request, $id)
    {
        $kurumId = $request->attributes->get('kurum_id');
        
        $request->validate([
            'durum' => 'required|in:aktif,pasif,tamamlandi,iptal'
        ]);

        $ilan = DB::table('kan_ilanlari')
            ->where('id', $id)
            ->where('kurum_id', $kurumId)
            ->first();

        if (!$ilan) {
            return redirect()->back()->with('error', 'İlan bulunamadı.');
        }

        DB::table('kan_ilanlari')
            ->where('id', $id)
            ->update([
                'durum' => $request->durum,
                'updated_at' => now()
            ]);

        return redirect()->back()->with('success', 'İlan durumu başarıyla güncellendi.');
    }

    public function create(Request $request)
    {
        $kurumId = $request->attributes->get('kurum_id');
        
        // Kullanıcılar listesi
        $kullanicilar = DB::table('kullanicilar')
            ->leftJoin('kan_gruplari', 'kullanicilar.kan_grubu_id', '=', 'kan_gruplari.id')
            ->where('kullanicilar.kurum_id', $kurumId)
            ->where('kullanicilar.aktif', true)
            ->select('kullanicilar.*', 'kan_gruplari.ad as kan_grubu')
            ->orderBy('kullanicilar.ad')
            ->get();

        // Kan grupları
        $kanGruplari = DB::table('kan_gruplari')->get();

        // İller
        $iller = DB::table('iller')->orderBy('ad')->get();
        if ($iller->isEmpty()) {
            $iller = DB::table('kurum_iller')->where('kurum_id', $kurumId)->orderBy('ad')->get();
        }

        return view('admin.kan-ilanlari.create', compact('kullanicilar', 'kanGruplari', 'iller'));
    }

    public function preview(Request $request)
    {
        $kurumId = $request->attributes->get('kurum_id');

        $validated = $request->validate([
            'kullanici_secim' => 'required|in:mevcut,yeni',
            'kullanici_id' => 'nullable|required_if:kullanici_secim,mevcut|exists:kullanicilar,id',
            'hizli_ad' => 'nullable|required_if:kullanici_secim,yeni|string|max:255',
            'hizli_soyad' => 'nullable|required_if:kullanici_secim,yeni|string|max:255',
            'hizli_telefon' => 'nullable|required_if:kullanici_secim,yeni|string|max:20',
            'hizli_email' => 'nullable|email|max:255',
            'hizli_kan_grubu_id' => 'nullable|required_if:kullanici_secim,yeni|exists:kan_gruplari,id',
            'hizli_tc_kimlik' => 'nullable|string|max:11',
            'hizli_il_id' => 'nullable|required_if:kullanici_secim,yeni|integer',
            'hizli_ilce_id' => 'nullable|required_if:kullanici_secim,yeni|integer',
            'kan_grubu_id' => 'required|exists:kan_gruplari,id',
            'unite_sayisi' => 'required|integer|min:1',
            'bagis_turu' => 'required|in:kan_unitesi,canli_bagisci',
            'ihtiyac_tarihi' => 'nullable|date',
            'ihtiyac_saati' => 'nullable|string|max:10',
            'hastane_adi' => 'required|string|max:255',
            'hastane_adresi' => 'required|string|max:500',
            'hastane_il' => 'required|string|max:255',
            'hastane_ilce' => 'required|string|max:255',
            'irtibat_telefonu' => 'required|string|max:20',
            'yakin_kisi_telefon' => 'nullable|string|max:20',
            'notlar' => 'nullable|string',
        ]);

        // Kullanıcı bilgilerini al
        $kullaniciBilgi = null;
        if ($validated['kullanici_secim'] == 'mevcut') {
            $kullaniciBilgi = DB::table('kullanicilar')
                ->leftJoin('kan_gruplari', 'kullanicilar.kan_grubu_id', '=', 'kan_gruplari.id')
                ->where('kullanicilar.id', $validated['kullanici_id'])
                ->select('kullanicilar.*', 'kan_gruplari.ad as kan_grubu')
                ->first();
        } else {
            $kanGrubuYeni = DB::table('kan_gruplari')->where('id', $validated['hizli_kan_grubu_id'])->first();
            $il = DB::table('iller')->where('id', $validated['hizli_il_id'])->first();
            $ilce = DB::table('ilceler')->where('id', $validated['hizli_ilce_id'])->first();
            
            $kullaniciBilgi = (object)[
                'ad' => $validated['hizli_ad'],
                'soyad' => $validated['hizli_soyad'],
                'telefon' => $validated['hizli_telefon'],
                'email' => $validated['hizli_email'],
                'kan_grubu' => $kanGrubuYeni ? $kanGrubuYeni->ad : '',
                'tc_kimlik' => $validated['hizli_tc_kimlik'] ?? '',
                'il' => $il ? $il->ad : '',
                'ilce' => $ilce ? $ilce->ad : '',
            ];
        }

        // Kan grubu bilgisi
        $kanGrubu = DB::table('kan_gruplari')->where('id', $validated['kan_grubu_id'])->first();

        return view('admin.kan-ilanlari.preview', compact('validated', 'kullaniciBilgi', 'kanGrubu'));
    }

    public function store(Request $request)
    {
        $kurumId = $request->attributes->get('kurum_id');

        $validated = $request->validate([
            'kullanici_secim' => 'required|in:mevcut,yeni',
            'kullanici_id' => 'nullable|required_if:kullanici_secim,mevcut|exists:kullanicilar,id',
            'hizli_ad' => 'nullable|required_if:kullanici_secim,yeni|string|max:255',
            'hizli_soyad' => 'nullable|required_if:kullanici_secim,yeni|string|max:255',
            'hizli_telefon' => 'nullable|required_if:kullanici_secim,yeni|string|max:20',
            'hizli_email' => 'nullable|email|max:255',
            'hizli_kan_grubu_id' => 'nullable|required_if:kullanici_secim,yeni|exists:kan_gruplari,id',
            'hizli_tc_kimlik' => 'nullable|string|max:11',
            'hizli_il_id' => 'nullable|required_if:kullanici_secim,yeni|integer',
            'hizli_ilce_id' => 'nullable|required_if:kullanici_secim,yeni|integer',
            'kan_grubu_id' => 'required|exists:kan_gruplari,id',
            'unite_sayisi' => 'required|integer|min:1',
            'bagis_turu' => 'required|in:kan_unitesi,canli_bagisci',
            'ihtiyac_tarihi' => 'nullable|date',
            'ihtiyac_saati' => 'nullable|string|max:10',
            'hastane_adi' => 'required|string|max:255',
            'hastane_adresi' => 'required|string|max:500',
            'hastane_il' => 'required|string|max:255',
            'hastane_ilce' => 'required|string|max:255',
            'irtibat_telefonu' => 'required|string|max:20',
            'yakin_kisi_telefon' => 'nullable|string|max:20',
            'notlar' => 'nullable|string',
        ]);

        // Kullanıcı ID'sini belirle veya oluştur
        $kullaniciId = null;
        $hastaAdSoyad = '';

        if ($validated['kullanici_secim'] == 'mevcut') {
            $kullaniciId = $validated['kullanici_id'];
            $kullanici = DB::table('kullanicilar')->where('id', $kullaniciId)->first();
            $hastaAdSoyad = $kullanici->ad . ' ' . $kullanici->soyad;
        } else {
            // İl ve ilçe ID'lerinin varlığını kontrol et (iller veya kurum_iller)
            if (isset($validated['hizli_il_id'])) {
                $ilExists = DB::table('iller')->where('id', $validated['hizli_il_id'])->exists() || 
                            DB::table('kurum_iller')->where('id', $validated['hizli_il_id'])->where('kurum_id', $kurumId)->exists();
                
                if (!$ilExists) {
                    return back()->withErrors(['hizli_il_id' => 'Seçilen il geçersiz.'])->withInput();
                }
            }
            
            if (isset($validated['hizli_ilce_id'])) {
                $ilceExists = DB::table('ilceler')->where('id', $validated['hizli_ilce_id'])->exists() || 
                             DB::table('kurum_ilceler')->where('id', $validated['hizli_ilce_id'])->exists();
                
                if (!$ilceExists) {
                    return back()->withErrors(['hizli_ilce_id' => 'Seçilen ilçe geçersiz.'])->withInput();
                }
            }
            
            // Hızlı kullanıcı oluştur
            $kullaniciId = DB::table('kullanicilar')->insertGetId([
                'kurum_id' => $kurumId,
                'ad' => $validated['hizli_ad'],
                'soyad' => $validated['hizli_soyad'],
                'email' => $validated['hizli_email'] ?? strtolower($validated['hizli_ad']) . rand(100, 999) . '@temp.com',
                'telefon' => $validated['hizli_telefon'],
                'tc_kimlik' => $validated['hizli_tc_kimlik'] ?? null,
                'il_id' => $validated['hizli_il_id'],
                'ilce_id' => $validated['hizli_ilce_id'],
                'kullanici_adi' => strtolower($validated['hizli_ad']) . rand(100, 999),
                'password' => bcrypt('123456'),
                'kan_grubu_id' => $validated['hizli_kan_grubu_id'],
                'aktif' => true,
                'email_verified_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            $hastaAdSoyad = $validated['hizli_ad'] . ' ' . $validated['hizli_soyad'];
        }

        // Benzersiz slug oluştur
        $slug = \Str::random(5);
        while (DB::table('kan_ilanlari')->where('slug', $slug)->exists()) {
            $slug = \Str::random(5);
        }

        // Link token oluştur
        $linkToken = \Str::random(5);
        while (DB::table('kan_ilanlari')->where('link_token', $linkToken)->exists()) {
            $linkToken = \Str::random(5);
        }

        // İlanı oluştur
        $ilanId = DB::table('kan_ilanlari')->insertGetId([
            'slug' => $slug,
            'kurum_id' => $kurumId,
            'kullanici_id' => $kullaniciId,
            'hasta_ad_soyad' => $hastaAdSoyad,
            'hastane_adi' => $validated['hastane_adi'],
            'hastane_adresi' => $validated['hastane_adresi'],
            'hastane_il' => $validated['hastane_il'],
            'hastane_ilce' => $validated['hastane_ilce'],
            'kan_grubu_id' => $validated['kan_grubu_id'],
            'unite_sayisi' => $validated['unite_sayisi'],
            'bagis_turu' => $validated['bagis_turu'],
            'tarih' => $validated['ihtiyac_tarihi'],
            'saat' => $validated['ihtiyac_saati'],
            'irtibat_telefonu' => $validated['irtibat_telefonu'],
            'yakin_kisi_telefon' => $validated['yakin_kisi_telefon'],
            'notlar' => $validated['notlar'],
            'durum' => 'aktif',
            'link_token' => $linkToken,
            'link_gecerlilik_tarihi' => now()->addHours(72),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // SMS gönderimi (kullanıcı ilanver sayfasındaki mantıkla)
        $ilan = DB::table('kan_ilanlari')->find($ilanId);
        $ilanLink = url('/ilan/' . $ilan->slug);
        $smsServisi = new SmsServisi();
        $smsSonuc = $smsServisi->ilanSmsGonder($kurumId, $ilan, $ilanLink);

        $flashMessage = 'Kan ilanı başarıyla oluşturuldu.';
        if ($smsSonuc['status'] === 'success') {
            $flashMessage .= ' ' . $smsSonuc['message'];
        } elseif ($smsSonuc['status'] === 'warning') {
            $flashMessage .= ' Ancak SMS gönderiminde bazı sorunlar yaşandı: ' . implode(', ', $smsSonuc['errors']);
        } elseif ($smsSonuc['status'] === 'error') {
            $flashMessage .= ' Ancak SMS gönderimi başarısız oldu: ' . $smsSonuc['message'];
        }

        return redirect()->route('admin.kan-ilanlari.show', $ilanId)->with('success', $flashMessage);
    }
}
