<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

class BolgeTanimlamaController extends Controller
{
    private function getKurumId(Request $request)
    {
        // This should be set by the KurumCozumleyici middleware
        $kurumId = $request->attributes->get('kurum_id');
        if (!$kurumId) {
            // Fallback or error for direct access without a resolved tenant
            // For now, let's try to find the first kurum for testing purposes.
            $kurum = DB::table('kurumlar')->first();
            if (!$kurum) {
                abort(404, 'Kurum bulunamadı.');
            }
            return $kurum->id;
        }
        return $kurumId;
    }

    public function index(Request $request)
    {
        $kurumId = $this->getKurumId($request);
        $iller = DB::table('kurum_iller')->where('kurum_id', $kurumId)->orderBy('ad')->get();
        $ilceler = DB::table('kurum_ilceler')->where('kurum_id', $kurumId)->orderBy('ad')->get()->groupBy('kurum_il_id');
        $beldeler = DB::table('kurum_beldeler')->where('kurum_id', $kurumId)->orderBy('ad')->get()->groupBy('kurum_ilce_id');

        return view('admin.bolge-tanimlamalari.index', compact('iller', 'ilceler', 'beldeler'));
    }

    public function create(Request $request)
    {
        $kurumId = $this->getKurumId($request);
        $tur = $request->query('tur', 'il');
        $parentId = $request->query('parent_id');
        
        $parent = null;
        $parentAd = null;
        
        if ($parentId) {
            if ($tur === 'ilce') {
                $parent = DB::table('kurum_iller')->where('id', $parentId)->where('kurum_id', $kurumId)->first();
                $parentAd = $parent ? $parent->ad : null;
            } elseif ($tur === 'belde') {
                $parent = DB::table('kurum_ilceler')->where('id', $parentId)->where('kurum_id', $kurumId)->first();
                $parentAd = $parent ? $parent->ad : null;
            }
        }
        
        return view('admin.bolge-tanimlamalari.create', compact('tur', 'parentId', 'parentAd'));
    }

    public function edit(Request $request, $id)
    {
        $kurumId = $this->getKurumId($request);
        $tur = $request->query('tur');
        
        $table = '';
        switch ($tur) {
            case 'il':
                $table = 'kurum_iller';
                break;
            case 'ilce':
                $table = 'kurum_ilceler';
                break;
            case 'belde':
                $table = 'kurum_beldeler';
                break;
        }
        
        $bolge = DB::table($table)->where('id', $id)->where('kurum_id', $kurumId)->first();
        
        if (!$bolge) {
            abort(404, 'Bölge bulunamadı');
        }
        
        return view('admin.bolge-tanimlamalari.edit', compact('bolge', 'tur', 'id'));
    }

    public function store(Request $request)
    {
        $kurumId = $this->getKurumId($request);
        $tur = $request->input('tur');
        $parentId = $request->input('parent_id');
        $ad = $request->input('ad');

        $validated = $request->validate([
            'ad' => 'required|string|max:255',
            'tur' => 'required|in:il,ilce,belde',
            'parent_id' => 'nullable|integer'
        ]);

        $data = [
            'kurum_id' => $kurumId,
            'ad' => $ad,
            'aktif_mi' => true,
            'created_at' => now(),
            'updated_at' => now(),
        ];

        $table = '';
        $parentCol = '';

        switch ($tur) {
            case 'il':
                $table = 'kurum_iller';
                break;
            case 'ilce':
                $table = 'kurum_ilceler';
                $parentCol = 'kurum_il_id';
                break;
            case 'belde':
                $table = 'kurum_beldeler';
                $parentCol = 'kurum_ilce_id';
                break;
        }

        if ($parentCol && $parentId) {
            $data[$parentCol] = $parentId;
        }

        // Unique validation
        $uniqueQuery = DB::table($table)->where('kurum_id', $kurumId)->where('ad', $ad);
        if($parentCol && $parentId) {
            $uniqueQuery->where($parentCol, $parentId);
        }

        if ($uniqueQuery->exists()) {
            return back()->with('error', 'Bu isimde bir kayıt zaten mevcut.');
        }

        DB::table($table)->insert($data);

        return redirect()->route('admin.bolge-tanimlamalari.index')->with('success', 'Bölge başarıyla eklendi.');
    }

    public function update(Request $request, $id)
    {
        $kurumId = $this->getKurumId($request);
        $tur = $request->input('tur');
        $ad = $request->input('ad');

        $request->validate([
            'ad' => 'required|string|max:255',
            'tur' => 'required|in:il,ilce,belde',
        ]);

        $table = '';
        if ($tur === 'il') $table = 'kurum_iller';
        if ($tur === 'ilce') $table = 'kurum_ilceler';
        if ($tur === 'belde') $table = 'kurum_beldeler';

        $item = DB::table($table)->where('id', $id)->where('kurum_id', $kurumId)->first();
        if(!$item) {
            return redirect()->route('admin.bolge-tanimlamalari.index')->with('error', 'Kayıt bulunamadı.');
        }

        $query = DB::table($table)->where('kurum_id', $kurumId)->where('ad', $ad)->where('id', '!=', $id);

        if(property_exists($item, 'kurum_il_id')) $query->where('kurum_il_id', $item->kurum_il_id);
        if(property_exists($item, 'kurum_ilce_id')) $query->where('kurum_ilce_id', $item->kurum_ilce_id);

        if($query->exists()){
            return back()->with('error', 'Bu isimde bir kayıt zaten mevcut.');
        }

        DB::table($table)->where('id', $id)->where('kurum_id', $kurumId)->update(['ad' => $ad, 'updated_at' => now()]);

        return redirect()->route('admin.bolge-tanimlamalari.index')->with('success', 'Bölge başarıyla güncellendi.');
    }

    public function destroy(Request $request, $id)
    {
        $kurumId = $this->getKurumId($request);
        $tur = $request->input('tur');

        $table = '';
        $childTable = '';
        $childFk = '';

        switch ($tur) {
            case 'il':
                $table = 'kurum_iller';
                $childTable = 'kurum_ilceler';
                $childFk = 'kurum_il_id';
                break;
            case 'ilce':
                $table = 'kurum_ilceler';
                $childTable = 'kurum_beldeler';
                $childFk = 'kurum_ilce_id';
                break;
            case 'belde':
                $table = 'kurum_beldeler';
                break;
        }

        if ($childTable && DB::table($childTable)->where($childFk, $id)->where('kurum_id', $kurumId)->exists()) {
            return back()->with('error', 'Bu kaydın altında başka kayıtlar olduğu için silinemez.');
        }

        DB::table($table)->where('id', $id)->where('kurum_id', $kurumId)->delete();

        return redirect()->route('admin.bolge-tanimlamalari.index')->with('success', 'Bölge başarıyla silindi.');
    }

    public function getParentOptions(Request $request)
    {
        $kurumId = $this->getKurumId($request);
        $tur = $request->input('tur');
        
        $table = '';
        switch ($tur) {
            case 'il':
                $table = 'kurum_iller';
                break;
            case 'ilce':
                $table = 'kurum_ilceler';
                break;
            case 'belde':
                $table = 'kurum_beldeler';
                break;
        }
        
        if (!$table) {
            return response()->json([]);
        }
        
        $options = DB::table($table)
            ->where('kurum_id', $kurumId)
            ->orderBy('ad')
            ->get(['id', 'ad']);
        
        return response()->json($options);
    }
}
